const url = new URL(location.href);
const channel = url.searchParams.get('c');
const uidParam = Number(url.searchParams.get('u')) || undefined; // optional
const callId = Number(url.searchParams.get('cid')) || null;

const token = localStorage.getItem('token');
if (!token || !channel) {
  alert('Missing auth or channel.');
  location.href = '/';
}

document.getElementById('channelInfo').textContent = `Channel: ${channel}`;

let client, localAudioTrack, localVideoTrack, remoteUid;
let appId;

async function getAppId() {
  const res = await fetch('/api/agora/config');
  const data = await res.json();
  return data.appId;
}

async function getToken(channelName, uid) {
  const res = await fetch('/api/agora/token', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json', Authorization: `Bearer ${token}` },
    body: JSON.stringify({ channelName, uid })
  });
  const data = await res.json();
  if (!res.ok) throw new Error(data.message || 'Failed to get token');
  return data.token;
}

async function start() {
  appId = await getAppId();
  client = AgoraRTC.createClient({ mode: 'rtc', codec: 'vp8' });

  const uid = uidParam || Number(JSON.parse(localStorage.getItem('user') || '{}').id);
  const rtcToken = await getToken(channel, uid);

  await client.join(appId, channel, rtcToken, uid);

  // Create tracks
  localAudioTrack = await AgoraRTC.createMicrophoneAudioTrack();
  localVideoTrack = await AgoraRTC.createCameraVideoTrack();

  // Play local
  localVideoTrack.play('local-player');

  // Publish tracks
  await client.publish([localAudioTrack, localVideoTrack]);

  // Handle remote users
  client.on('user-published', async (user, mediaType) => {
    await client.subscribe(user, mediaType);
    remoteUid = user.uid;
    if (mediaType === 'video') {
      const remoteContainer = document.getElementById('remote-player');
      user.videoTrack.play(remoteContainer);
    }
    if (mediaType === 'audio') {
      user.audioTrack.play();
    }
  });

  client.on('user-unpublished', (user) => {
    if (user.uid === remoteUid) {
      document.getElementById('remote-player').innerHTML = '';
    }
  });
}

start().catch(err => {
  console.error(err);
  alert('Failed to start call: ' + err.message);
  location.href = '/dashboard.html';
});

// Controls
let micOn = true, camOn = true;

document.getElementById('btnToggleMic').onclick = async () => {
  if (!localAudioTrack) return;
  micOn = !micOn;
  await localAudioTrack.setEnabled(micOn);
  document.getElementById('btnToggleMic').classList.toggle('btn-outline-light', micOn);
  document.getElementById('btnToggleMic').classList.toggle('btn-warning', !micOn);
};

document.getElementById('btnToggleCam').onclick = async () => {
  if (!localVideoTrack) return;
  camOn = !camOn;
  await localVideoTrack.setEnabled(camOn);
  document.getElementById('btnToggleCam').classList.toggle('btn-outline-light', camOn);
  document.getElementById('btnToggleCam').classList.toggle('btn-warning', !camOn);
};

async function leaveCall() {
  try {
    localAudioTrack && localAudioTrack.close();
    localVideoTrack && localVideoTrack.close();
    await client.unpublish();
    await client.leave();
  } catch {}
}

const socket = io('/', { auth: { token } });

async function endCall() {
  if (callId) socket.emit('call:end', { callId });
}

window.addEventListener('beforeunload', endCall);

document.getElementById('btnLeave').onclick = async () => {
  try { await leaveCall(); } finally { await endCall(); location.href = '/dashboard.html'; }
};
