const API = '/api';
const token = localStorage.getItem('token');
const user = JSON.parse(localStorage.getItem('user') || 'null');

if (!token) window.location.href = '/';

document.getElementById('navbarUser').textContent = user ? `@${user.username}` : '';

document.getElementById('logoutBtn').addEventListener('click', () => {
  localStorage.removeItem('token');
  localStorage.removeItem('user');
  window.location.href = '/';
});

// --- Socket.IO signaling ---
const socket = io('/', { auth: { token } });
socket.on('connect_error', (err) => console.warn('socket error', err.message));

// Search users as you type (debounced)
const searchInput = document.getElementById('searchInput');
const searchResults = document.getElementById('searchResults');
let t;
searchInput.addEventListener('input', () => {
  clearTimeout(t);
  t = setTimeout(async () => {
    const q = searchInput.value.trim();
    if (q.length < 2) { searchResults.innerHTML = ''; return; }
    const res = await fetch(`${API}/users?q=${encodeURIComponent(q)}`, { headers: { Authorization: `Bearer ${token}` }});
    const data = await res.json();
    renderSearchResults(data);
  }, 250);
});

function renderSearchResults(list) {
  searchResults.innerHTML = '';
  if (!list.length) {
    searchResults.innerHTML = '<div class="text-muted">No users found</div>';
    return;
  }
  list.forEach(u => {
    const div = document.createElement('div');
    div.className = 'd-flex align-items-center justify-content-between border rounded p-2';
    div.innerHTML = `<div>@${u.username}</div>`;
    const btn = document.createElement('button');
    btn.className = 'btn btn-sm btn-outline-primary';
    btn.textContent = 'Add to contacts';
    btn.onclick = async () => {
      const res = await fetch(`${API}/contacts/${u.id}`, { method: 'POST', headers: { Authorization: `Bearer ${token}` }});
      if (res.ok) { loadContacts(); btn.disabled = true; btn.textContent = 'Added'; }
    };
    div.appendChild(btn);
    searchResults.appendChild(div);
  });
}

// Contacts
const contactsTableBody = document.querySelector('#contactsTable tbody');
const contactCount = document.getElementById('contactCount');

async function loadContacts() {
  const res = await fetch(`${API}/contacts`, { headers: { Authorization: `Bearer ${token}` }});
  const data = await res.json();
  contactsTableBody.innerHTML = '';
  data.forEach(c => {
    const tr = document.createElement('tr');
    tr.dataset.id = c.id;
    const tdUser = document.createElement('td');
    const badge = document.createElement('span');
    badge.className = 'badge rounded-pill bg-secondary me-2 presence-badge';
    badge.textContent = 'offline';
    tdUser.appendChild(badge);
    tdUser.appendChild(document.createTextNode(`@${c.username}`));

    const tdAction = document.createElement('td');
    tdAction.className = 'text-end';
    const callBtn = document.createElement('button');
    callBtn.className = 'btn btn-sm btn-success';
    callBtn.textContent = 'Call';
    callBtn.onclick = () => socket.emit('call:invite', { calleeId: c.id });
    tdAction.appendChild(callBtn);

    tr.appendChild(tdUser);
    tr.appendChild(tdAction);
    contactsTableBody.appendChild(tr);
  });
  contactCount.textContent = data.length;
  refreshPresence();
}

function refreshPresence() {
  const ids = Array.from(document.querySelectorAll('#contactsTable tbody tr')).map(tr => Number(tr.dataset.id)).filter(Boolean);
  socket.emit('presence:list', ids);
}

socket.on('presence:update', ({ online }) => {
  const set = new Set(online);
  document.querySelectorAll('#contactsTable tbody tr').forEach(tr => {
    const badge = tr.querySelector('.presence-badge');
    const id = Number(tr.dataset.id);
    if (set.has(id)) {
      badge.classList.remove('bg-secondary');
      badge.classList.add('bg-success');
      badge.textContent = 'online';
    } else {
      badge.classList.remove('bg-success');
      badge.classList.add('bg-secondary');
      badge.textContent = 'offline';
    }
  });
});

// Ringing modal handlers
let incoming = null;
const incomingModal = new bootstrap.Modal('#incomingModal');
const ringtone = document.getElementById('ringtone');

socket.on('call:ring', ({ callId, channel, from }) => {
  incoming = { callId, channel, from };
  document.getElementById('incomingFrom').textContent = `@${from.username} is calling…`;
  if (ringtone) { ringtone.currentTime = 0; ringtone.play().catch(()=>{}); }
  incomingModal.show();

  // Foreground notification
  if (Notification.permission === 'granted') {
    const link = `${location.origin}/call.html?c=${encodeURIComponent(channel)}&u=${user.id}&cid=${callId}`;
    if (navigator.serviceWorker?.ready) {
      navigator.serviceWorker.ready.then(reg => reg.showNotification('Incoming call', { body: `@${from.username}`, data: { url: link } }));
    } else {
      const n = new Notification('Incoming call', { body: `@${from.username}` });
      n.onclick = () => window.open(link, '_blank');
    }
  }
});

document.getElementById('btnAccept').onclick = () => {
  if (!incoming) return;
  socket.emit('call:accept', { callId: incoming.callId });
  if (ringtone) ringtone.pause();
  const url = `${location.origin}/call.html?c=${encodeURIComponent(incoming.channel)}&u=${user.id}&cid=${incoming.callId}`;
  location.href = url;
};

document.getElementById('btnReject').onclick = () => {
  if (!incoming) return;
  socket.emit('call:reject', { callId: incoming.callId });
  if (ringtone) ringtone.pause();
  incoming = null;
};

socket.on('call:outgoing', ({ callId, channel }) => {
  const url = `${location.origin}/call.html?c=${encodeURIComponent(channel)}&u=${user.id}&cid=${callId}`;
  location.href = url;
});

// Push subscription setup
async function setupNotifications() {
  if (!('serviceWorker' in navigator) || !('PushManager' in window)) return;
  try {
    const reg = await navigator.serviceWorker.register('/sw.js');
    if (Notification.permission === 'default') {
      await Notification.requestPermission();
    }
    const resp = await fetch('/api/push/vapid-public');
    const { publicKey } = await resp.json();
    if (!publicKey) return; // not configured
    const sub = await reg.pushManager.getSubscription() || await reg.pushManager.subscribe({
      userVisibleOnly: true,
      applicationServerKey: urlBase64ToUint8Array(publicKey)
    });
    await fetch('/api/push/subscribe', {
      method: 'POST', headers: { 'Content-Type': 'application/json', Authorization: `Bearer ${token}` },
      body: JSON.stringify(sub)
    });
  } catch (e) { console.warn('push setup failed', e); }
}
function urlBase64ToUint8Array(base64String) {
  const padding = '='.repeat((4 - base64String.length % 4) % 4);
  const base64 = (base64String + padding).replace(/-/g, '+').replace(/_/g, '/');
  const raw = atob(base64); const arr = new Uint8Array(raw.length);
  for (let i = 0; i < raw.length; ++i) arr[i] = raw.charCodeAt(i);
  return arr;
}

setupNotifications();
loadContacts();
