import 'dotenv/config';
import express from 'express';
import cors from 'cors';
import path from 'path';
import { fileURLToPath } from 'url';
import http from 'http';
import { Server as SocketIOServer } from 'socket.io';
import jwt from 'jsonwebtoken';

import { pool } from './db.js';
import authRoutes from './routes/auth.js';
import userRoutes from './routes/users.js';
import contactRoutes from './routes/contacts.js';
import agoraRoutes from './routes/agora.js';
import callsRoutes from './routes/calls.js';
import adminRoutes from './routes/admin.js';
import pushRoutes, { notifyUser } from './routes/push.js';

const app = express();
const PORT = process.env.PORT || 3000;

// Paths
const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);
const publicDir = path.join(__dirname, '..', 'public');

// Middleware
app.use(cors({ origin: process.env.CORS_ORIGIN?.split(',') || true }));
app.use(express.json());
app.use(express.urlencoded({ extended: true }));

// Static frontend
app.use(express.static(publicDir));

// API routes (prefix /api)
app.use('/api/auth', authRoutes);
app.use('/api/users', userRoutes);
app.use('/api/contacts', contactRoutes);
app.use('/api/agora', agoraRoutes);
app.use('/api/calls', callsRoutes);
app.use('/api/admin', adminRoutes);
app.use('/api/push', pushRoutes);

// Health check
app.get('/api/health', (req, res) => res.json({ ok: true }));

const server = http.createServer(app);
const io = new SocketIOServer(server, {
  cors: { origin: process.env.CORS_ORIGIN?.split(',') || true }
});

// In-memory maps (consider Redis for scale)
const userToSocket = new Map(); // userId -> socket.id
const socketToUser = new Map(); // socket.id -> { id, username }

io.use((socket, next) => {
  const { token } = socket.handshake.auth || {};
  if (!token) return next(new Error('no token'));
  try {
    const payload = jwt.verify(token, process.env.JWT_SECRET);
    socket.user = payload; // { id, email, username }
    next();
  } catch (e) {
    next(new Error('bad token'));
  }
});

io.on('connection', (socket) => {
  const u = socket.user;
  userToSocket.set(u.id, socket.id);
  socketToUser.set(socket.id, u);
  socket.join(`user:${u.id}`);

  socket.emit('presence:me', { online: true });

  socket.on('presence:list', async (contactIds = []) => {
    const online = contactIds.filter((id) => userToSocket.has(id));
    socket.emit('presence:update', { online });
  });

  // call invitation -> ringing
  socket.on('call:invite', async ({ calleeId }) => {
    const calleeSocketId = userToSocket.get(calleeId);
    const channel = `vc_${u.id}_${calleeId}_${Date.now()}`;

    // create a call record with status 'ringing'
    const [result] = await pool.query(
      'INSERT INTO calls (caller_id, callee_id, channel, status) VALUES (?, ?, ?, ?)',
      [u.id, calleeId, channel, 'ringing']
    );
    const callId = result.insertId;

    if (calleeSocketId) {
      io.to(calleeSocketId).emit('call:ring', {
        callId, channel, from: { id: u.id, username: u.username }
      });
      socket.emit('call:outgoing', { callId, channel, ringing: true });
    } else {
      // callee offline -> mark missed
      await pool.query('UPDATE calls SET status = ? WHERE id = ?', ['missed', callId]);
      socket.emit('call:unavailable', { callId, reason: 'offline' });
    }

    // Push notification (if configured)
    await notifyUser(calleeId, {
      title: 'Incoming call',
      body: `@${u.username} is calling you`,
      data: { url: `/call.html?c=${encodeURIComponent(channel)}&u=${calleeId}&cid=${callId}` }
    });
  });

  socket.on('call:accept', async ({ callId }) => {
    const [rows] = await pool.query('SELECT * FROM calls WHERE id = ?', [callId]);
    if (!rows.length) return;
    const call = rows[0];
    if (call.status !== 'ringing') return;

    await pool.query('UPDATE calls SET status = ?, started_at = NOW() WHERE id = ?', ['active', callId]);

    const callerSock = userToSocket.get(call.caller_id);
    io.to(callerSock).emit('call:accepted', { callId, channel: call.channel });
    io.to(socket.id).emit('call:accepted', { callId, channel: call.channel });
  });

  socket.on('call:reject', async ({ callId }) => {
    await pool.query('UPDATE calls SET status = ? WHERE id = ?', ['rejected', callId]);
    const [rows] = await pool.query('SELECT caller_id FROM calls WHERE id = ?', [callId]);
    if (rows.length) {
      const callerSock = userToSocket.get(rows[0].caller_id);
      callerSock && io.to(callerSock).emit('call:rejected', { callId });
    }
  });

  socket.on('call:end', async ({ callId }) => {
    await pool.query(
      'UPDATE calls SET status = ?, ended_at = NOW(), duration_seconds = TIMESTAMPDIFF(SECOND, started_at, NOW()) WHERE id = ?',
      ['ended', callId]
    );

    const [rows] = await pool.query('SELECT caller_id, callee_id FROM calls WHERE id = ?', [callId]);
    if (rows.length) {
      const { caller_id, callee_id } = rows[0];
      const a = userToSocket.get(caller_id);
      const b = userToSocket.get(callee_id);
      a && io.to(a).emit('call:ended', { callId });
      b && io.to(b).emit('call:ended', { callId });
    }
  });

  socket.on('disconnect', () => {
    userToSocket.delete(u.id);
    socketToUser.delete(socket.id);
  });
});

server.listen(PORT, () => {
  console.log(`Server running on http://localhost:${PORT}`);
});
